@extends('layouts.app')

@section('title', 'WhatsApp Management')

@push('style')
<style>
    .whatsapp-card {
        border-radius: 15px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        border: none;
    }
    .whatsapp-header {
        background: linear-gradient(135deg, #25d366 0%, #128c7e 100%);
        color: white;
        border-radius: 15px 15px 0 0;
        padding: 20px;
    }
    .country-code-selector {
        background: #f8f9fa;
        border: 1px solid #ced4da;
        border-radius: 5px 0 0 5px;
        padding: 8px 12px;
        font-weight: 600;
        color: #495057;
    }
    .phone-input {
        border-radius: 0 5px 5px 0;
        border-left: none;
    }
    .message-template-card {
        background: #f8f9fa;
        border: 1px solid #e9ecef;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 15px;
        transition: all 0.3s ease;
    }
    .message-template-card:hover {
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        border-color: #25d366;
    }
    .message-preview {
        background: #dcf8c6;
        border-radius: 18px;
        padding: 12px 16px;
        margin: 10px 0;
        font-family: 'Segoe UI', sans-serif;
        font-size: 14px;
        white-space: pre-line;
        position: relative;
        box-shadow: 0 1px 2px rgba(0,0,0,0.1);
    }
    .message-preview::after {
        content: '';
        position: absolute;
        right: -8px;
        bottom: 10px;
        border-left: 8px solid #dcf8c6;
        border-top: 8px solid transparent;
        border-bottom: 8px solid transparent;
    }
    .api-status {
        display: inline-block;
        padding: 4px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
    }
    .api-status.connected {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    .api-status.disconnected {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
    .variable-helper {
        background: #e7f3ff;
        border: 1px solid #b6d7ff;
        border-radius: 5px;
        padding: 10px;
        margin-bottom: 10px;
    }
    .variable-tag {
        background: #007bff;
        color: white;
        border-radius: 4px;
        padding: 2px 8px;
        margin: 2px;
        display: inline-block;
        font-size: 11px;
        cursor: pointer;
        transition: all 0.2s ease;
    }
    .variable-tag:hover {
        background: #0056b3;
        transform: translateY(-1px);
    }
    .toggle-switch {
        position: relative;
        display: inline-block;
        width: 60px;
        height: 34px;
    }
    .toggle-switch input {
        opacity: 0;
        width: 0;
        height: 0;
    }
    .slider {
        position: absolute;
        cursor: pointer;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: #ccc;
        transition: .4s;
        border-radius: 34px;
    }
    .slider:before {
        position: absolute;
        content: "";
        height: 26px;
        width: 26px;
        left: 4px;
        bottom: 4px;
        background-color: white;
        transition: .4s;
        border-radius: 50%;
    }
    input:checked + .slider {
        background-color: #25d366;
    }
    input:checked + .slider:before {
        transform: translateX(26px);
    }
</style>
@endpush

@section('main')
<div class="content-header">
    <div class="container-fluid">
        <div class="row mb-2">
            <div class="col-sm-6">
                <h1 class="m-0">WhatsApp Management</h1>
            </div>
            <div class="col-sm-6">
                <ol class="breadcrumb float-sm-right">
                    <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">Home</a></li>
                    <li class="breadcrumb-item"><a href="{{ route('invoice-settings.index') }}">Invoice Settings</a></li>
                    <li class="breadcrumb-item active">WhatsApp Management</li>
                </ol>
            </div>
        </div>
    </div>
</div>

<section class="content">
    <div class="container-fluid">
        @if(session('success'))
            <div class="alert alert-success alert-dismissible fade show">
                <i class="fab fa-whatsapp mr-2"></i>{{ session('success') }}
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        @endif

        <form action="{{ route('invoice-settings.update') }}" method="POST" id="whatsappForm">
            @csrf
            @method('PUT')
            
            <!-- Include existing settings as hidden fields -->
            <input type="hidden" name="company_name" value="{{ $settings->company_name ?? 'Your Company Name' }}">
            <input type="hidden" name="invoice_header_text" value="{{ $settings->invoice_header_text ?? 'INVOICE' }}">
            <input type="hidden" name="invoice_footer_text" value="{{ $settings->invoice_footer_text ?? '' }}">
            <input type="hidden" name="invoice_template" value="{{ $settings->invoice_template ?? 'default' }}">
            <input type="hidden" name="invoice_terms_conditions" value="{{ $settings->invoice_terms_conditions ?? '' }}">
            
            <div class="row">
                <!-- Connection & QR Settings -->
                <div class="col-md-6">
                    <div class="card whatsapp-card mb-4">
                        <div class="whatsapp-header">
                            <h4 class="mb-0">
                                <i class="fab fa-whatsapp mr-2"></i>WhatsApp QR Code System
                                <span class="api-status" id="connectionStatus">
                                    <i class="fas fa-spinner fa-spin"></i> Checking...
                                </span>
                            </h4>
                        </div>
                        <div class="card-body">
                            <!-- Connection Status Alert -->
                            <div class="alert" id="connectionAlert">
                                <div id="alertContent">
                                    <i class="fas fa-spinner fa-spin mr-2"></i>Checking WhatsApp connection...
                                </div>
                            </div>

                            <div class="form-group">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <label class="mb-0 font-weight-bold">Enable WhatsApp QR Integration</label>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="whatsapp_enabled" value="1" 
                                               {{ ($settings->whatsapp_enabled ?? true) ? 'checked' : '' }}>
                                        <span class="slider"></span>
                                    </label>
                                </div>
                                <small class="text-muted">Generate QR codes for easy WhatsApp messaging</small>
                            </div>

                            <div class="form-group">
                                <label for="whatsapp_country_code">Default Country Code</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <select class="form-control country-code-selector" name="whatsapp_country_code" id="whatsapp_country_code">
                                            <!-- South Asia -->
                                            <option value="+880" {{ ($settings->whatsapp_country_code ?? '+880') == '+880' ? 'selected' : '' }}>🇧🇩 +880 (Bangladesh)</option>
                                            <option value="+91" {{ ($settings->whatsapp_country_code ?? '') == '+91' ? 'selected' : '' }}>🇮🇳 +91 (India)</option>
                                            <option value="+92" {{ ($settings->whatsapp_country_code ?? '') == '+92' ? 'selected' : '' }}>🇵🇰 +92 (Pakistan)</option>
                                            <option value="+94" {{ ($settings->whatsapp_country_code ?? '') == '+94' ? 'selected' : '' }}>�� +94 (Sri Lanka)</option>
                                            <option value="+977" {{ ($settings->whatsapp_country_code ?? '') == '+977' ? 'selected' : '' }}>🇳🇵 +977 (Nepal)</option>
                                            <option value="+975" {{ ($settings->whatsapp_country_code ?? '') == '+975' ? 'selected' : '' }}>🇧🇹 +975 (Bhutan)</option>
                                            <option value="+960" {{ ($settings->whatsapp_country_code ?? '') == '+960' ? 'selected' : '' }}>🇲🇻 +960 (Maldives)</option>
                                            
                                            <!-- Southeast Asia -->
                                            <option value="+60" {{ ($settings->whatsapp_country_code ?? '') == '+60' ? 'selected' : '' }}>🇲🇾 +60 (Malaysia)</option>
                                            <option value="+62" {{ ($settings->whatsapp_country_code ?? '') == '+62' ? 'selected' : '' }}>🇮🇩 +62 (Indonesia)</option>
                                            <option value="+63" {{ ($settings->whatsapp_country_code ?? '') == '+63' ? 'selected' : '' }}>🇵🇭 +63 (Philippines)</option>
                                            <option value="+65" {{ ($settings->whatsapp_country_code ?? '') == '+65' ? 'selected' : '' }}>�🇸🇬 +65 (Singapore)</option>
                                            <option value="+66" {{ ($settings->whatsapp_country_code ?? '') == '+66' ? 'selected' : '' }}>🇹🇭 +66 (Thailand)</option>
                                            <option value="+84" {{ ($settings->whatsapp_country_code ?? '') == '+84' ? 'selected' : '' }}>🇻🇳 +84 (Vietnam)</option>
                                            <option value="+855" {{ ($settings->whatsapp_country_code ?? '') == '+855' ? 'selected' : '' }}>🇰🇭 +855 (Cambodia)</option>
                                            <option value="+856" {{ ($settings->whatsapp_country_code ?? '') == '+856' ? 'selected' : '' }}>🇱🇦 +856 (Laos)</option>
                                            <option value="+95" {{ ($settings->whatsapp_country_code ?? '') == '+95' ? 'selected' : '' }}>🇲� +95 (Myanmar)</option>
                                            <option value="+673" {{ ($settings->whatsapp_country_code ?? '') == '+673' ? 'selected' : '' }}>🇧🇳 +673 (Brunei)</option>
                                            
                                            <!-- East Asia -->
                                            <option value="+86" {{ ($settings->whatsapp_country_code ?? '') == '+86' ? 'selected' : '' }}>🇨🇳 +86 (China)</option>
                                            <option value="+81" {{ ($settings->whatsapp_country_code ?? '') == '+81' ? 'selected' : '' }}>🇯🇵 +81 (Japan)</option>
                                            <option value="+82" {{ ($settings->whatsapp_country_code ?? '') == '+82' ? 'selected' : '' }}>🇰🇷 +82 (South Korea)</option>
                                            <option value="+852" {{ ($settings->whatsapp_country_code ?? '') == '+852' ? 'selected' : '' }}>🇭🇰 +852 (Hong Kong)</option>
                                            <option value="+853" {{ ($settings->whatsapp_country_code ?? '') == '+853' ? 'selected' : '' }}>🇲🇴 +853 (Macau)</option>
                                            <option value="+886" {{ ($settings->whatsapp_country_code ?? '') == '+886' ? 'selected' : '' }}>🇹🇼 +886 (Taiwan)</option>
                                            <option value="+976" {{ ($settings->whatsapp_country_code ?? '') == '+976' ? 'selected' : '' }}>🇲🇳 +976 (Mongolia)</option>
                                            
                                            <!-- West Asia / Middle East -->
                                            <option value="+971" {{ ($settings->whatsapp_country_code ?? '') == '+971' ? 'selected' : '' }}>🇦🇪 +971 (UAE)</option>
                                            <option value="+966" {{ ($settings->whatsapp_country_code ?? '') == '+966' ? 'selected' : '' }}>🇸🇦 +966 (Saudi Arabia)</option>
                                            <option value="+965" {{ ($settings->whatsapp_country_code ?? '') == '+965' ? 'selected' : '' }}>🇰🇼 +965 (Kuwait)</option>
                                            <option value="+974" {{ ($settings->whatsapp_country_code ?? '') == '+974' ? 'selected' : '' }}>🇶🇦 +974 (Qatar)</option>
                                            <option value="+973" {{ ($settings->whatsapp_country_code ?? '') == '+973' ? 'selected' : '' }}>🇧🇭 +973 (Bahrain)</option>
                                            <option value="+968" {{ ($settings->whatsapp_country_code ?? '') == '+968' ? 'selected' : '' }}>🇴🇲 +968 (Oman)</option>
                                            <option value="+98" {{ ($settings->whatsapp_country_code ?? '') == '+98' ? 'selected' : '' }}>🇮🇷 +98 (Iran)</option>
                                            <option value="+90" {{ ($settings->whatsapp_country_code ?? '') == '+90' ? 'selected' : '' }}>🇹🇷 +90 (Turkey)</option>
                                            <option value="+962" {{ ($settings->whatsapp_country_code ?? '') == '+962' ? 'selected' : '' }}>�� +962 (Jordan)</option>
                                            <option value="+961" {{ ($settings->whatsapp_country_code ?? '') == '+961' ? 'selected' : '' }}>🇱🇧 +961 (Lebanon)</option>
                                            
                                            <!-- Central Asia -->
                                            <option value="+7" {{ ($settings->whatsapp_country_code ?? '') == '+7' ? 'selected' : '' }}>🇰🇿 +7 (Kazakhstan)</option>
                                            <option value="+996" {{ ($settings->whatsapp_country_code ?? '') == '+996' ? 'selected' : '' }}>�🇬 +996 (Kyrgyzstan)</option>
                                            <option value="+998" {{ ($settings->whatsapp_country_code ?? '') == '+998' ? 'selected' : '' }}>🇺🇿 +998 (Uzbekistan)</option>
                                            <option value="+993" {{ ($settings->whatsapp_country_code ?? '') == '+993' ? 'selected' : '' }}>🇹🇲 +993 (Turkmenistan)</option>
                                            <option value="+992" {{ ($settings->whatsapp_country_code ?? '') == '+992' ? 'selected' : '' }}>🇹🇯 +992 (Tajikistan)</option>
                                            
                                            <!-- Additional Middle East Countries -->
                                            <option value="+967" {{ ($settings->whatsapp_country_code ?? '') == '+967' ? 'selected' : '' }}>🇾🇪 +967 (Yemen)</option>
                                            <option value="+964" {{ ($settings->whatsapp_country_code ?? '') == '+964' ? 'selected' : '' }}>🇮🇶 +964 (Iraq)</option>
                                            <option value="+963" {{ ($settings->whatsapp_country_code ?? '') == '+963' ? 'selected' : '' }}>🇸🇾 +963 (Syria)</option>
                                            <option value="+972" {{ ($settings->whatsapp_country_code ?? '') == '+972' ? 'selected' : '' }}>🇮🇱 +972 (Israel)</option>
                                            <option value="+970" {{ ($settings->whatsapp_country_code ?? '') == '+970' ? 'selected' : '' }}>🇵🇸 +970 (Palestine)</option>
                                            <option value="+995" {{ ($settings->whatsapp_country_code ?? '') == '+995' ? 'selected' : '' }}>🇬🇪 +995 (Georgia)</option>
                                            <option value="+374" {{ ($settings->whatsapp_country_code ?? '') == '+374' ? 'selected' : '' }}>🇦🇲 +374 (Armenia)</option>
                                            <option value="+994" {{ ($settings->whatsapp_country_code ?? '') == '+994' ? 'selected' : '' }}>🇦🇿 +994 (Azerbaijan)</option>
                                            <option value="+357" {{ ($settings->whatsapp_country_code ?? '') == '+357' ? 'selected' : '' }}>🇨🇾 +357 (Cyprus)</option>
                                            
                                            <!-- Additional Southeast Asian Countries -->
                                            <option value="+670" {{ ($settings->whatsapp_country_code ?? '') == '+670' ? 'selected' : '' }}>🇹🇱 +670 (East Timor)</option>
                                            
                                            <!-- Additional South Asian Countries -->
                                            <option value="+93" {{ ($settings->whatsapp_country_code ?? '') == '+93' ? 'selected' : '' }}>🇦🇫 +93 (Afghanistan)</option>
                                            
                                            <!-- Additional East Asian Countries -->
                                            <option value="+850" {{ ($settings->whatsapp_country_code ?? '') == '+850' ? 'selected' : '' }}>🇰🇵 +850 (North Korea)</option>
                                            
                                            <!-- Russia (covers Asian part too) -->
                                            <option value="+7" {{ ($settings->whatsapp_country_code ?? '') == '+7' ? 'selected' : '' }}>🇷🇺 +7 (Russia)</option>
                                            
                                            <!-- Popular Other Countries -->
                                            <option value="+1" {{ ($settings->whatsapp_country_code ?? '') == '+1' ? 'selected' : '' }}>🇺🇸 +1 (USA/Canada)</option>
                                            <option value="+44" {{ ($settings->whatsapp_country_code ?? '') == '+44' ? 'selected' : '' }}>🇬🇧 +44 (UK)</option>
                                            <option value="+49" {{ ($settings->whatsapp_country_code ?? '') == '+49' ? 'selected' : '' }}>🇩🇪 +49 (Germany)</option>
                                            <option value="+33" {{ ($settings->whatsapp_country_code ?? '') == '+33' ? 'selected' : '' }}>🇫🇷 +33 (France)</option>
                                            <option value="+39" {{ ($settings->whatsapp_country_code ?? '') == '+39' ? 'selected' : '' }}>🇮🇹 +39 (Italy)</option>
                                            <option value="+34" {{ ($settings->whatsapp_country_code ?? '') == '+34' ? 'selected' : '' }}>🇪🇸 +34 (Spain)</option>
                                            <option value="+61" {{ ($settings->whatsapp_country_code ?? '') == '+61' ? 'selected' : '' }}>🇦🇺 +61 (Australia)</option>
                                            <option value="+55" {{ ($settings->whatsapp_country_code ?? '') == '+55' ? 'selected' : '' }}>🇧🇷 +55 (Brazil)</option>
                                            <option value="+27" {{ ($settings->whatsapp_country_code ?? '') == '+27' ? 'selected' : '' }}>🇿🇦 +27 (South Africa)</option>
                                            <option value="+20" {{ ($settings->whatsapp_country_code ?? '') == '+20' ? 'selected' : '' }}>🇪🇬 +20 (Egypt)</option>
                                            
                                            <!-- Additional European Countries -->
                                            <option value="+31" {{ ($settings->whatsapp_country_code ?? '') == '+31' ? 'selected' : '' }}>🇳🇱 +31 (Netherlands)</option>
                                            <option value="+32" {{ ($settings->whatsapp_country_code ?? '') == '+32' ? 'selected' : '' }}>🇧🇪 +32 (Belgium)</option>
                                            <option value="+41" {{ ($settings->whatsapp_country_code ?? '') == '+41' ? 'selected' : '' }}>🇨🇭 +41 (Switzerland)</option>
                                            <option value="+43" {{ ($settings->whatsapp_country_code ?? '') == '+43' ? 'selected' : '' }}>🇦🇹 +43 (Austria)</option>
                                            <option value="+48" {{ ($settings->whatsapp_country_code ?? '') == '+48' ? 'selected' : '' }}>🇵🇱 +48 (Poland)</option>
                                            <option value="+420" {{ ($settings->whatsapp_country_code ?? '') == '+420' ? 'selected' : '' }}>🇨🇿 +420 (Czech Republic)</option>
                                            <option value="+421" {{ ($settings->whatsapp_country_code ?? '') == '+421' ? 'selected' : '' }}>🇸🇰 +421 (Slovakia)</option>
                                            <option value="+36" {{ ($settings->whatsapp_country_code ?? '') == '+36' ? 'selected' : '' }}>🇭🇺 +36 (Hungary)</option>
                                            <option value="+40" {{ ($settings->whatsapp_country_code ?? '') == '+40' ? 'selected' : '' }}>🇷🇴 +40 (Romania)</option>
                                            <option value="+359" {{ ($settings->whatsapp_country_code ?? '') == '+359' ? 'selected' : '' }}>🇧🇬 +359 (Bulgaria)</option>
                                            <option value="+30" {{ ($settings->whatsapp_country_code ?? '') == '+30' ? 'selected' : '' }}>🇬🇷 +30 (Greece)</option>
                                            <option value="+351" {{ ($settings->whatsapp_country_code ?? '') == '+351' ? 'selected' : '' }}>🇵🇹 +351 (Portugal)</option>
                                            <option value="+46" {{ ($settings->whatsapp_country_code ?? '') == '+46' ? 'selected' : '' }}>🇸🇪 +46 (Sweden)</option>
                                            <option value="+47" {{ ($settings->whatsapp_country_code ?? '') == '+47' ? 'selected' : '' }}>🇳🇴 +47 (Norway)</option>
                                            <option value="+45" {{ ($settings->whatsapp_country_code ?? '') == '+45' ? 'selected' : '' }}>🇩🇰 +45 (Denmark)</option>
                                            <option value="+358" {{ ($settings->whatsapp_country_code ?? '') == '+358' ? 'selected' : '' }}>🇫🇮 +358 (Finland)</option>
                                            <option value="+380" {{ ($settings->whatsapp_country_code ?? '') == '+380' ? 'selected' : '' }}>🇺🇦 +380 (Ukraine)</option>
                                            <option value="+375" {{ ($settings->whatsapp_country_code ?? '') == '+375' ? 'selected' : '' }}>🇧🇾 +375 (Belarus)</option>
                                            <option value="+372" {{ ($settings->whatsapp_country_code ?? '') == '+372' ? 'selected' : '' }}>🇪🇪 +372 (Estonia)</option>
                                            <option value="+371" {{ ($settings->whatsapp_country_code ?? '') == '+371' ? 'selected' : '' }}>🇱🇻 +371 (Latvia)</option>
                                            <option value="+370" {{ ($settings->whatsapp_country_code ?? '') == '+370' ? 'selected' : '' }}>🇱🇹 +370 (Lithuania)</option>
                                            <option value="+353" {{ ($settings->whatsapp_country_code ?? '') == '+353' ? 'selected' : '' }}>🇮🇪 +353 (Ireland)</option>
                                            <option value="+354" {{ ($settings->whatsapp_country_code ?? '') == '+354' ? 'selected' : '' }}>🇮🇸 +354 (Iceland)</option>
                                            
                                            <!-- Additional African Countries -->
                                            <option value="+234" {{ ($settings->whatsapp_country_code ?? '') == '+234' ? 'selected' : '' }}>🇳🇬 +234 (Nigeria)</option>
                                            <option value="+254" {{ ($settings->whatsapp_country_code ?? '') == '+254' ? 'selected' : '' }}>🇰🇪 +254 (Kenya)</option>
                                            <option value="+233" {{ ($settings->whatsapp_country_code ?? '') == '+233' ? 'selected' : '' }}>🇬🇭 +233 (Ghana)</option>
                                            <option value="+255" {{ ($settings->whatsapp_country_code ?? '') == '+255' ? 'selected' : '' }}>🇹🇿 +255 (Tanzania)</option>
                                            <option value="+256" {{ ($settings->whatsapp_country_code ?? '') == '+256' ? 'selected' : '' }}>🇺🇬 +256 (Uganda)</option>
                                            <option value="+212" {{ ($settings->whatsapp_country_code ?? '') == '+212' ? 'selected' : '' }}>🇲🇦 +212 (Morocco)</option>
                                            <option value="+213" {{ ($settings->whatsapp_country_code ?? '') == '+213' ? 'selected' : '' }}>🇩🇿 +213 (Algeria)</option>
                                            <option value="+216" {{ ($settings->whatsapp_country_code ?? '') == '+216' ? 'selected' : '' }}>🇹🇳 +216 (Tunisia)</option>
                                            <option value="+218" {{ ($settings->whatsapp_country_code ?? '') == '+218' ? 'selected' : '' }}>🇱🇾 +218 (Libya)</option>
                                            <option value="+251" {{ ($settings->whatsapp_country_code ?? '') == '+251' ? 'selected' : '' }}>🇪🇹 +251 (Ethiopia)</option>
                                            
                                            <!-- North American Countries -->
                                            <option value="+52" {{ ($settings->whatsapp_country_code ?? '') == '+52' ? 'selected' : '' }}>🇲🇽 +52 (Mexico)</option>
                                            
                                            <!-- South American Countries -->
                                            <option value="+54" {{ ($settings->whatsapp_country_code ?? '') == '+54' ? 'selected' : '' }}>🇦🇷 +54 (Argentina)</option>
                                            <option value="+56" {{ ($settings->whatsapp_country_code ?? '') == '+56' ? 'selected' : '' }}>🇨🇱 +56 (Chile)</option>
                                            <option value="+57" {{ ($settings->whatsapp_country_code ?? '') == '+57' ? 'selected' : '' }}>🇨🇴 +57 (Colombia)</option>
                                            <option value="+51" {{ ($settings->whatsapp_country_code ?? '') == '+51' ? 'selected' : '' }}>🇵🇪 +51 (Peru)</option>
                                            <option value="+58" {{ ($settings->whatsapp_country_code ?? '') == '+58' ? 'selected' : '' }}>🇻🇪 +58 (Venezuela)</option>
                                            <option value="+593" {{ ($settings->whatsapp_country_code ?? '') == '+593' ? 'selected' : '' }}>🇪🇨 +593 (Ecuador)</option>
                                            <option value="+595" {{ ($settings->whatsapp_country_code ?? '') == '+595' ? 'selected' : '' }}>🇵🇾 +595 (Paraguay)</option>
                                            <option value="+598" {{ ($settings->whatsapp_country_code ?? '') == '+598' ? 'selected' : '' }}>🇺🇾 +598 (Uruguay)</option>
                                            
                                            <!-- Oceania Countries -->
                                            <option value="+64" {{ ($settings->whatsapp_country_code ?? '') == '+64' ? 'selected' : '' }}>🇳🇿 +64 (New Zealand)</option>
                                            <option value="+679" {{ ($settings->whatsapp_country_code ?? '') == '+679' ? 'selected' : '' }}>🇫🇯 +679 (Fiji)</option>
                                        </select>
                                    </div>
                                    <input type="text" class="form-control phone-input" placeholder="1712345678" readonly>
                                </div>
                                <small class="text-muted">This will be used as default when customer phone numbers don't include country code</small>
                            </div>

                            <div class="form-group">
                                <div class="d-flex justify-content-between align-items-center">
                                    <label class="mb-0">Auto-format Phone Numbers</label>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="whatsapp_auto_format_numbers" value="1" 
                                               {{ ($settings->whatsapp_auto_format_numbers ?? true) ? 'checked' : '' }}>
                                        <span class="slider"></span>
                                    </label>
                                </div>
                                <small class="text-muted">Automatically add country code to local numbers</small>
                            </div>

                            <div class="alert alert-info">
                                <i class="fas fa-qrcode mr-2"></i>
                                <strong>QR Code System:</strong> No API setup needed! Generate QR codes that customers can scan to send WhatsApp messages directly.
                            </div>

                            <!-- QR Code Demo -->
                            <div class="card bg-light">
                                <div class="card-header">
                                    <h6 class="mb-0"><i class="fas fa-qrcode mr-2"></i>QR Code Demo</h6>
                                </div>
                                <div class="card-body text-center">
                                    <img id="demoQr" src="https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=https://wa.me/8801712345678?text=Hello%20from%20your%20mobile%20store!" 
                                         alt="Demo QR Code" class="img-fluid mb-2" style="max-width: 150px;">
                                    <p class="small text-muted mb-0">Sample QR code for customer messaging</p>
                                    <button type="button" class="btn btn-sm btn-primary mt-2" onclick="generateDemoQr()">
                                        <i class="fas fa-refresh mr-1"></i>Generate New Demo
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- QR Code Settings -->
                    <div class="card whatsapp-card">
                        <div class="card-header bg-info">
                            <h5 class="mb-0"><i class="fas fa-cogs mr-2"></i>QR Code Settings</h5>
                        </div>
                        <div class="card-body">
                            <div class="form-group">
                                <label for="qr_code_size">QR Code Size</label>
                                <select class="form-control" id="qr_code_size" name="qr_code_size">
                                    <option value="150">150x150 (Small)</option>
                                    <option value="200" selected>200x200 (Medium)</option>
                                    <option value="300">300x300 (Large)</option>
                                    <option value="400">400x400 (Extra Large)</option>
                                </select>
                                <small class="text-muted">Size of generated QR codes in pixels</small>
                            </div>

                            <div class="form-group">
                                <label for="qr_provider">QR Code Provider</label>
                                <select class="form-control" id="qr_provider" name="qr_provider">
                                    <option value="qrserver" selected>QR Server (Recommended)</option>
                                    <option value="google">Google Charts API</option>
                                </select>
                                <small class="text-muted">Service used to generate QR codes</small>
                            </div>

                            <div class="form-group">
                                <div class="d-flex justify-content-between align-items-center">
                                    <label class="mb-0">Show QR Code in Invoices</label>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="show_qr_in_invoice" value="1" checked>
                                        <span class="slider"></span>
                                    </label>
                                </div>
                                <small class="text-muted">Include QR code in printed/PDF invoices</small>
                            </div>

                            <div class="form-group">
                                <div class="d-flex justify-content-between align-items-center">
                                    <label class="mb-0">Auto-show QR after Sale</label>
                                    <label class="toggle-switch">
                                        <input type="checkbox" name="auto_show_qr" value="1" 
                                               {{ ($settings->auto_send_whatsapp ?? false) ? 'checked' : '' }}>
                                        <span class="slider"></span>
                                    </label>
                                </div>
                                <small class="text-muted">Automatically display QR code popup after completing sale</small>
                            </div>
                        </div>
                    </div>

                    <!-- Live Testing Section -->
                    <div class="card whatsapp-card mt-4">
                        <div class="card-header bg-warning">
                            <h5 class="mb-0"><i class="fas fa-test-tube mr-2"></i>Live Testing</h5>
                        </div>
                        <div class="card-body">
                            <div class="form-group">
                                <label for="test_phone">Test Phone Number</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text" id="test-country-code">+880</span>
                                    </div>
                                    <input type="text" class="form-control" id="test_phone" 
                                           placeholder="1712345678" value="1712345678">
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="test_message">Test Message</label>
                                <textarea class="form-control" id="test_message" rows="3" 
                                          placeholder="Enter your test message here...">🎉 Test message from {{ $settings->company_name ?? 'Your Mobile Store' }}!

This is a test WhatsApp message to verify the integration is working properly. 📱

Thank you!</textarea>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <button type="button" class="btn btn-success btn-block" onclick="testWhatsAppUrl()">
                                        <i class="fab fa-whatsapp mr-2"></i>Test WhatsApp URL
                                    </button>
                                </div>
                                <div class="col-md-6">
                                    <button type="button" class="btn btn-info btn-block" onclick="generateTestQr()">
                                        <i class="fas fa-qrcode mr-2"></i>Generate Test QR
                                    </button>
                                </div>
                            </div>

                            <!-- Test Results -->
                            <div class="mt-3" id="testResults" style="display: none;">
                                <div class="card bg-light">
                                    <div class="card-body">
                                        <h6><i class="fas fa-link mr-2"></i>Generated WhatsApp URL:</h6>
                                        <div class="input-group mb-3">
                                            <input type="text" class="form-control" id="generatedUrl" readonly>
                                            <div class="input-group-append">
                                                <button class="btn btn-outline-secondary" type="button" onclick="copyToClipboard('generatedUrl')">
                                                    <i class="fas fa-copy"></i>
                                                </button>
                                            </div>
                                        </div>
                                        
                                        <h6><i class="fas fa-qrcode mr-2"></i>QR Code:</h6>
                                        <div class="text-center">
                                            <img id="testQrImage" src="" alt="Test QR Code" class="img-fluid" style="max-width: 200px; display: none;">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Message Templates -->
                <div class="col-md-6">
                    <div class="card whatsapp-card mb-4">
                        <div class="card-header bg-success">
                            <h5 class="mb-0"><i class="fas fa-comment-alt mr-2"></i>Message Templates</h5>
                        </div>
                        <div class="card-body">
                            <div class="variable-helper">
                                <strong>Available Variables:</strong><br>
                                <span class="variable-tag" data-variable="{company_name}">Company Name</span>
                                <span class="variable-tag" data-variable="{customer_name}">Customer Name</span>
                                <span class="variable-tag" data-variable="{customer_phone}">Customer Phone</span>
                                <span class="variable-tag" data-variable="{invoice_number}">Invoice Number</span>
                                <span class="variable-tag" data-variable="{invoice_date}">Invoice Date</span>
                                <span class="variable-tag" data-variable="{due_date}">Due Date</span>
                                <span class="variable-tag" data-variable="{item_list}">Item List</span>
                                <span class="variable-tag" data-variable="{currency}">Currency</span>
                                <span class="variable-tag" data-variable="{total_amount}">Total Amount</span>
                                <span class="variable-tag" data-variable="{paid_amount}">Paid Amount</span>
                                <span class="variable-tag" data-variable="{due_amount}">Due Amount</span>
                                <span class="variable-tag" data-variable="{customer_total_due}">Customer Total Due</span>
                                <span class="variable-tag" data-variable="{payment_method}">Payment Method</span>
                                <span class="variable-tag" data-variable="{payment_status}">Payment Status</span>
                                <span class="variable-tag" data-variable="{invoice_link}">Invoice Link</span>
                            </div>

                            <!-- Welcome Message -->
                            <div class="message-template-card">
                                <h6><i class="fas fa-handshake text-primary mr-2"></i>Welcome Message</h6>
                                <textarea class="form-control message-template" name="whatsapp_welcome_message" 
                                          rows="4" placeholder="Welcome message for new customers...">{{ $settings->whatsapp_welcome_message ?? '' }}</textarea>
                                <div class="message-preview" id="welcomePreview"></div>
                            </div>

                            <!-- Invoice Message -->
                            <div class="message-template-card">
                                <h6><i class="fas fa-receipt text-primary mr-2"></i>Invoice Message</h6>
                                <textarea class="form-control message-template" name="whatsapp_invoice_message_template" 
                                          rows="8" placeholder="Invoice message template...">{{ $settings->whatsapp_invoice_message_template ?? '🧾 Invoice #{invoice_number}

👤 Customer: {customer_name}
📞 Phone: {customer_phone}
📅 Date: {invoice_date}

📦 Items:
{item_list}

💰 Total Amount: {currency} {total_amount}
💳 Payment Method: {payment_method}
📊 Payment Status: {payment_status}

🙏 Thank you for your business!

🔗 View full invoice: {invoice_link}' }}</textarea>
                                <div class="message-preview" id="invoicePreview"></div>
                            </div>

                            <!-- Payment Reminder -->
                            <div class="message-template-card">
                                <h6><i class="fas fa-credit-card text-warning mr-2"></i>Payment Reminder</h6>
                                <textarea class="form-control message-template" name="whatsapp_payment_reminder_message" 
                                          rows="4" placeholder="Payment reminder message...">{{ $settings->whatsapp_payment_reminder_message ?? '' }}</textarea>
                                <div class="message-preview" id="reminderPreview"></div>
                            </div>

                            <!-- Delivery Message -->
                            <div class="message-template-card">
                                <h6><i class="fas fa-truck text-info mr-2"></i>Delivery Message</h6>
                                <textarea class="form-control message-template" name="whatsapp_delivery_message" 
                                          rows="4" placeholder="Delivery notification message...">{{ $settings->whatsapp_delivery_message ?? '' }}</textarea>
                                <div class="message-preview" id="deliveryPreview"></div>
                            </div>

                            <!-- Thank You Message -->
                            <div class="message-template-card">
                                <h6><i class="fas fa-heart text-success mr-2"></i>Thank You Message</h6>
                                <textarea class="form-control message-template" name="whatsapp_thank_you_message" 
                                          rows="4" placeholder="Thank you message after purchase...">{{ $settings->whatsapp_thank_you_message ?? '' }}</textarea>
                                <div class="message-preview" id="thankYouPreview"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-body text-center">
                            <button type="submit" class="btn btn-success btn-lg mr-3">
                                <i class="fab fa-whatsapp mr-2"></i>Save WhatsApp Settings
                            </button>
                            <button type="button" class="btn btn-info btn-lg mr-3" onclick="testQrCode()">
                                <i class="fas fa-qrcode mr-2"></i>Test QR Code
                            </button>
                            <button type="button" class="btn btn-warning btn-lg mr-3" onclick="resetToDefaults()">
                                <i class="fas fa-undo mr-2"></i>Reset to Defaults
                            </button>
                            <a href="{{ route('invoice-settings.index') }}" class="btn btn-secondary btn-lg">
                                <i class="fas fa-arrow-left mr-2"></i>Back to Invoice Settings
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</section>
@endsection

@push('scripts')
<script>
$(document).ready(function() {
    // Check connection status on page load
    checkWhatsAppConnection();
    
    // Update previews on page load
    updateAllPreviews();

    // Update previews when templates change
    $('.message-template').on('input', function() {
        updateAllPreviews();
    });

    // Variable tag click to insert
    $('.variable-tag').click(function() {
        const variable = $(this).data('variable');
        const activeTextarea = document.activeElement;
        
        if (activeTextarea.tagName === 'TEXTAREA') {
            const cursorPos = activeTextarea.selectionStart;
            const textBefore = activeTextarea.value.substring(0, cursorPos);
            const textAfter = activeTextarea.value.substring(cursorPos);
            
            activeTextarea.value = textBefore + variable + textAfter;
            activeTextarea.selectionStart = activeTextarea.selectionEnd = cursorPos + variable.length;
            activeTextarea.focus();
            
            updateAllPreviews();
        }
    });

    function checkWhatsAppConnection() {
        // For web-based WhatsApp, we don't need API connection
        // Just show that the system is ready
        setTimeout(function() {
            updateConnectionStatus('web-ready');
        }, 1000);
    }

    function updateConnectionStatus(status, deviceInfo = null) {
        const statusSpan = $('#connectionStatus');
        const alertDiv = $('#connectionAlert');
        const alertContent = $('#alertContent');
        
        switch(status) {
            case 'web-ready':
                statusSpan.removeClass().addClass('api-status connected float-right')
                    .html('<i class="fas fa-globe"></i> Web Ready');
                
                alertDiv.removeClass().addClass('alert alert-success');
                alertContent.html(`
                    <i class="fas fa-check-circle mr-2"></i>
                    <strong>WhatsApp Web System Ready!</strong> Your system is configured to use WhatsApp Web URLs and QR codes.
                    <div class="mt-2">
                        <button class="btn btn-sm btn-success mr-2" onclick="testWhatsAppUrl()">
                            <i class="fab fa-whatsapp mr-1"></i>Test Now
                        </button>
                        <button class="btn btn-sm btn-info" onclick="generateTestQr()">
                            <i class="fas fa-qrcode mr-1"></i>Generate QR
                        </button>
                    </div>
                `);
                break;
                
            case 'connected':
                statusSpan.removeClass().addClass('api-status connected float-right')
                    .html('<i class="fas fa-check-circle"></i> Connected');
                
                alertDiv.removeClass().addClass('alert alert-success');
                alertContent.html(`
                    <i class="fas fa-check-circle mr-2"></i>
                    <strong>WhatsApp Connected!</strong> Your device "${deviceInfo?.device_name}" is connected.
                    <a href="{{ route('whatsapp.connect') }}" class="btn btn-sm btn-outline-success ml-2">
                        <i class="fas fa-cogs mr-1"></i>Manage Connection
                    </a>
                `);
                break;
                
            case 'disconnected':
                statusSpan.removeClass().addClass('api-status disconnected float-right')
                    .html('<i class="fas fa-times-circle"></i> Disconnected');
                
                alertDiv.removeClass().addClass('alert alert-warning');
                alertContent.html(`
                    <i class="fas fa-exclamation-triangle mr-2"></i>
                    <strong>WhatsApp Not Connected</strong> - Connect your WhatsApp to send messages directly from your phone.
                    <a href="{{ route('whatsapp.connect') }}" class="btn btn-sm btn-warning ml-2">
                        <i class="fas fa-qrcode mr-1"></i>Connect Now
                    </a>
                `);
                break;
                
            case 'error':
                statusSpan.removeClass().addClass('api-status disconnected float-right')
                    .html('<i class="fas fa-exclamation-circle"></i> Error');
                
                alertDiv.removeClass().addClass('alert alert-danger');
                alertContent.html(`
                    <i class="fas fa-exclamation-circle mr-2"></i>
                    <strong>Connection Error</strong> - Unable to check WhatsApp status.
                    <button class="btn btn-sm btn-danger ml-2" onclick="checkWhatsAppConnection()">
                        <i class="fas fa-refresh mr-1"></i>Retry
                    </button>
                `);
                break;
        }
    }

    // Make checkWhatsAppConnection available globally
    window.checkWhatsAppConnection = checkWhatsAppConnection;

    function updateAllPreviews() {
        updatePreview('whatsapp_welcome_message', 'welcomePreview');
        updatePreview('whatsapp_invoice_message_template', 'invoicePreview');
        updatePreview('whatsapp_payment_reminder_message', 'reminderPreview');
        updatePreview('whatsapp_delivery_message', 'deliveryPreview');
        updatePreview('whatsapp_thank_you_message', 'thankYouPreview');
    }

    function updatePreview(templateName, previewId) {
        let template = $(`textarea[name="${templateName}"]`).val();
        
        if (!template) {
            $(`#${previewId}`).hide();
            return;
        }

        // Replace variables with sample data
        template = template.replace(/{company_name}/g, 'Mobile Store Pro');
        template = template.replace(/{customer_name}/g, 'mahbub');
        template = template.replace(/{customer_phone}/g, '8801612384172');
        template = template.replace(/{invoice_number}/g, '43');
        template = template.replace(/{invoice_date}/g, '10/20/2025');
        template = template.replace(/{due_date}/g, new Date(Date.now() + 30*24*60*60*1000).toLocaleDateString());
        template = template.replace(/{item_list}/g, '• Apple iPhone Xs amx x1 @ BDT 935.00 = BDT 935.00');
        template = template.replace(/{currency}/g, 'BDT');
        template = template.replace(/{total_amount}/g, '935.00');
        template = template.replace(/{paid_amount}/g, '635.00');
        template = template.replace(/{due_amount}/g, '300.00');
        template = template.replace(/{customer_total_due}/g, '1,250.00');
        template = template.replace(/{payment_method}/g, 'credit');
        template = template.replace(/{payment_status}/g, 'partial');
        template = template.replace(/{invoice_link}/g, 'http://127.0.0.1:8000/sales/43/table-invoice');
        
        $(`#${previewId}`).text(template).show();
    }
});

function generateDemoQr() {
    const countryCode = $('#whatsapp_country_code').val();
    const phone = countryCode.replace('+', '') + '1712345678';
    const message = encodeURIComponent('Hello! I would like to inquire about your mobile phones. 📱');
    const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=https://wa.me/${phone}?text=${message}`;
    
    $('#demoQr').attr('src', qrUrl);
    toastr.success('Demo QR code updated with current country code!');
}

function testQrCode() {
    // Create a test QR code popup
    const countryCode = $('#whatsapp_country_code').val();
    const phone = countryCode.replace('+', '') + '1712345678';
    const message = encodeURIComponent('🎉 Test message from ' + ($('#company_name').val() || 'Your Mobile Store') + '!\n\nThis is a test QR code to verify WhatsApp integration is working properly. 📱\n\nThank you!');
    const whatsappUrl = `https://wa.me/${phone}?text=${message}`;
    const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(whatsappUrl)}`;
    
    // Create modal
    const modal = `
        <div class="modal fade" id="qrTestModal" tabindex="-1">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header bg-success text-white">
                        <h5 class="modal-title">
                            <i class="fab fa-whatsapp mr-2"></i>Test QR Code
                        </h5>
                        <button type="button" class="close text-white" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body text-center">
                        <img src="${qrUrl}" alt="Test QR Code" class="img-fluid mb-3" style="max-width: 300px;">
                        <p class="lead">Scan this QR code with your phone to test!</p>
                        <p class="text-muted">
                            <strong>Phone:</strong> ${phone}<br>
                            <strong>Country:</strong> ${$('#whatsapp_country_code option:selected').text()}
                        </p>
                        <a href="${whatsappUrl}" target="_blank" class="btn btn-success">
                            <i class="fab fa-whatsapp mr-2"></i>Open WhatsApp Web
                        </a>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                        <button type="button" class="btn btn-primary" onclick="downloadQr('${qrUrl}')">
                            <i class="fas fa-download mr-2"></i>Download QR
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal and add new one
    $('#qrTestModal').remove();
    $('body').append(modal);
    $('#qrTestModal').modal('show');
}

function downloadQr(qrUrl) {
    const link = document.createElement('a');
    link.href = qrUrl;
    link.download = 'whatsapp-qr-test.png';
    link.click();
    toastr.success('QR code download started!');
}

// Live testing functions
function testWhatsAppUrl() {
    const countryCode = $('#whatsapp_country_code').val();
    const phoneNumber = $('#test_phone').val();
    const message = $('#test_message').val();
    
    if (!phoneNumber) {
        toastr.error('Please enter a phone number for testing');
        return;
    }
    
    if (!message.trim()) {
        toastr.error('Please enter a test message');
        return;
    }
    
    // Format phone number
    const fullPhone = countryCode.replace('+', '') + phoneNumber;
    
    // Generate WhatsApp URL
    const whatsappUrl = `https://api.whatsapp.com/send/?phone=${fullPhone}&text=${encodeURIComponent(message)}`;
    
    // Show results
    $('#generatedUrl').val(whatsappUrl);
    $('#testResults').show();
    
    // Open WhatsApp
    window.open(whatsappUrl, '_blank');
    
    toastr.success('WhatsApp opened successfully! Test message URL generated.');
}

function generateTestQr() {
    const countryCode = $('#whatsapp_country_code').val();
    const phoneNumber = $('#test_phone').val();
    const message = $('#test_message').val();
    
    if (!phoneNumber) {
        toastr.error('Please enter a phone number for testing');
        return;
    }
    
    if (!message.trim()) {
        toastr.error('Please enter a test message');
        return;
    }
    
    // Format phone number and create URL
    const fullPhone = countryCode.replace('+', '') + phoneNumber;
    const whatsappUrl = `https://api.whatsapp.com/send/?phone=${fullPhone}&text=${encodeURIComponent(message)}`;
    
    // Generate QR code
    const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(whatsappUrl)}`;
    
    // Show results
    $('#generatedUrl').val(whatsappUrl);
    $('#testQrImage').attr('src', qrUrl).show();
    $('#testResults').show();
    
    toastr.success('Test QR code generated successfully!');
}

function copyToClipboard(elementId) {
    const element = document.getElementById(elementId);
    element.select();
    element.setSelectionRange(0, 99999); // For mobile devices
    
    try {
        document.execCommand('copy');
        toastr.success('URL copied to clipboard!');
    } catch (err) {
        toastr.error('Failed to copy URL');
    }
}

// Update test country code when main country code changes
$(document).ready(function() {
    $('#whatsapp_country_code').on('change', function() {
        $('#test-country-code').text($(this).val());
    });
});

function resetToDefaults() {
    if (confirm('Are you sure you want to reset all WhatsApp templates to defaults? This cannot be undone.')) {
        // Reset all message templates to empty (will trigger defaults from controller)
        $('textarea[name="whatsapp_welcome_message"]').val('');
        $('textarea[name="whatsapp_payment_reminder_message"]').val('');
        $('textarea[name="whatsapp_delivery_message"]').val('');
        $('textarea[name="whatsapp_thank_you_message"]').val('');
        
        // Reset other settings
        $('#whatsapp_country_code').val('+880');
        $('#whatsapp_business_hours').val('9:00 AM - 6:00 PM');
        $('#qr_code_size').val('200');
        $('#qr_provider').val('qrserver');
        
        // Update previews and demo
        updateAllPreviews();
        generateDemoQr();
        
        toastr.info('Templates reset to defaults. Don\'t forget to save!');
    }
}
</script>
@endpush